// Game music emulator common interface

// Game_Music_Emu 0.4.0
#ifndef MUSIC_EMU_H
#define MUSIC_EMU_H

#include "gme.h"
#include "blargg_common.h"
#include "Data_Reader.h"
class Multi_Buffer;

struct Music_Emu {
public:
// Basic functionality

	// Set output sample rate. Must be called only once before loading file.
	blargg_err_t set_sample_rate( long sample_rate );
	
	// Load music from file or custom reader
	blargg_err_t load_file( const char* path );
	virtual blargg_err_t load( Data_Reader& ) = 0;
	
	// Start a track, where 0 is the first track
	void start_track( int );
	
	// Generate 'count' samples info 'buf'. Output is in stereo.
	typedef short sample_t;
	void play( long count, sample_t* buf );
	
// Informational
	
	// Sample rate sound is generated at
	long sample_rate() const;
	
	// Number of voices used by currently loaded file
	int voice_count() const;
	
	// Names of voices
	virtual const char** voice_names() const;
	
	// Number of tracks or 0 if no file has been loaded
	int track_count() const;
	
	// Information for a track (length, name, author, etc.)
	virtual blargg_err_t track_info( track_info_t* out, int track ) const;
	blargg_err_t track_info( track_info_t* out ) const; // info for current track
	
	// Index of current track or -1 if one hasn't been started
	int current_track() const;
	
// Track status/control

	// Number of milliseconds played since beginning of track
	long tell() const;
	
	// Seek to new time in track
	void seek( long msec );
	
	// Skip count samples
	void skip( long count );
	
	// True if a track was started and has since ended. Currently only logged
	// format tracks (VGM, GYM) without loop points have an ending.
	bool track_ended() const;
	
	// Set start time and length of track fade out. Once fade ends track_ended() returns
	// true. Fade time can be changed while track is playing.
	void set_fade( long start_msec, long length_msec = 8000 );
	
	// Disable automatic end-of-track detection and skipping of silence at beginning
	void disable_silence_detection( bool = true );
	
	// Number of errors encountered while playing track due to undefined CPU
	// instructions in emulated formats and undefined stream events in
	// logged formats.
	int error_count() const;
	
// Sound customization

	// Request use of custom multichannel buffer. Only supported by "classic" emulators;
	// on others this has no effect. Should be called only once *before* set_sample_rate().
	virtual void set_buffer( Multi_Buffer* ) { }
	
	// Mute voice n if bit n (1 << n) of mask is set
	virtual void mute_voices( int mask );
	
	// Frequency equalizer parameters (see gme.txt)
	struct equalizer_t {
		double treble; // -50.0 = muffled, 0 = flat, +5.0 = extra-crisp
		long   bass;   // 1 = full bass, 90 = average, 16000 = almost no bass
	};
	
	// Current frequency equalizater parameters
	const equalizer_t& equalizer() const;
	
	// Set frequency equalizer parameters
	virtual void set_equalizer( equalizer_t const& );
	
	// Equalizer settings for TV speaker
	static equalizer_t const tv_eq;
	
public:
	Music_Emu();
	virtual ~Music_Emu();
	BLARGG_DISABLE_NOTHROW
protected:
	typedef BOOST::uint8_t byte;
	void set_voice_count( int n ) { voice_count_ = n; }
	void set_track_count( int n ) { track_count_ = n; }
	void set_track_ended() { emu_track_ended_ = true; }
	void log_error( int n = 1 ) { error_count_ += n; }
	void remute_voices();
	void unload();
	
	virtual blargg_err_t set_sample_rate_( long sample_rate ) = 0;
	virtual void start_track_( int ) = 0;
	virtual void play_( long count, sample_t* out ) = 0;
	virtual void skip_( long count );
private:
	// noncopyable
	Music_Emu( const Music_Emu& );
	Music_Emu& operator = ( const Music_Emu& );
	
	// general
	equalizer_t equalizer_;
	long sample_rate_;
	int voice_count_;
	int mute_mask_;
	int track_count_;
	long msec_to_samples( long msec ) const;
	
	// track-specific
	int current_track_;
	int error_count_;
	long out_time;      // number of samples played since start of track
	long emu_time;      // number of samples emulator has generated since start of track
	bool emu_track_ended_; // emulator has reached end of track
	volatile bool track_ended_;
	void set_track_vars( int track );
	
	// fading
	long fade_start;
	int fade_step;
	void handle_fade( long count, sample_t* out );
	
	// silence detection
	bool silence_detection;
	long silence_time;  // number of samples where most recent silence began
	long silence_count; // number of samples of silence to play before using buf
	long buf_remain;// number of samples left in silence buffer
	enum { buf_size = 2048 };
	blargg_vector<sample_t> buf;
	void fill_buf();
	void emu_play( long count, sample_t* out );
};

struct gme_type_t_
{
	Music_Emu* (*new_emu)();
	blargg_err_t (*read_track_info)( Data_Reader&, track_info_t*, int track );
};

#define GME_COPY_FIELD( in, out, name ) \
	{ out->copy_field( out->name, in.name, sizeof in.name ); }

inline int Music_Emu::error_count() const   { return error_count_; }
inline int Music_Emu::voice_count() const   { return voice_count_; }
inline int Music_Emu::track_count() const   { return track_count_; }
inline bool Music_Emu::track_ended() const  { return track_ended_; }
inline void Music_Emu::mute_voices( int mask ) { mute_mask_ = mask; }
inline void Music_Emu::remute_voices() { mute_voices( mute_mask_ ); }
inline const Music_Emu::equalizer_t& Music_Emu::equalizer() const { return equalizer_; }
inline void Music_Emu::set_equalizer( const equalizer_t& eq ) { equalizer_ = eq; }
inline long Music_Emu::sample_rate() const { return sample_rate_; }
inline void Music_Emu::disable_silence_detection( bool b ) { silence_detection = !b; }
inline int Music_Emu::current_track() const { return current_track_; }

inline void Music_Emu::start_track_( int ) { }

Music_Emu* gme_new_( Music_Emu*, long sample_rate );

#endif
